/* Simple udisks based automounter
 *
 * (C) 2010 Canonical Ltd.
 * Author: Martin Pitt <martin.pitt@ubuntu.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using UDisks;
using DBus;

const string[] gui_clients = { "Thunar", "nautilus", "pcmanfm", "emelfm2" };

DBus.Connection dbus;
bool coldplug = true;

async void
on_device_added_async(ObjectPath device)
{
    /* do not open file managers during coldplug */
    /* need to do this before yield */
    bool open_client = !coldplug;

    UDisks.Device dev = (UDisks.Device) dbus.get_object("org.freedesktop.UDisks",
	    device);
    stdout.printf("Device added: %s (%s)\n", device, dev.device_file);

    if (dev.id_usage == "filesystem" && !dev.device_is_mounted && 
	!dev.device_is_system_internal && !dev.device_presentation_hide && 
	!dev.device_presentation_nopolicy) 
    {
	stdout.printf("  appropriate for auto-mounting\n");
	string path = yield dev.filesystem_mount("", {"auth_no_user_interaction"});
	stdout.printf("  mounted to: %s\n", path);

	/* now launch a GUI window on the mount path */
	if (open_client) {
	    int pid;
	    foreach (string client in gui_clients) {
		stdout.printf("  trying to launch %s\n", client);
		try {
		    if (Process.spawn_async(null, {client, path}, null,
			    SpawnFlags.SEARCH_PATH, null, out pid))
			stdout.printf("  success!\n");
		    break;
		} catch { /* ignore, try next one */ }
	    }
	}
    }
}

void
on_device_added(ObjectPath device)
{
    on_device_added_async.begin(device);
}

void
on_device_removed(ObjectPath device)
{
    stdout.printf("Device removed: %s\n", device);
}

int 
main(string[] args)
{
    UDisks.Daemon udisks;

    /* Get UDisks D-Bus proxy */
    try {
	dbus = DBus.Bus.get (DBus.BusType.SYSTEM);
	udisks = (UDisks.Daemon) dbus.get_object("org.freedesktop.UDisks",
		"/org/freedesktop/UDisks");
    } catch (DBus.Error e) {
	stderr.printf ("Could not connect to UDisks: %s\n", e.message);
	return 1;
    }

    stdout.printf("Connected to UDisks %s\n", udisks.daemon_version);

    /* Coldplug */
    foreach (ObjectPath o in udisks.enumerate_devices())
	on_device_added_async (o);
    coldplug = false;

    udisks.device_added.connect (on_device_added);
    udisks.device_removed.connect (on_device_removed);

    new MainLoop().run();

    return 0;
}

