/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include "mwb-crash-page.h"
#include "mwb-aspect-clone.h"
#include "mwb-mozembed.h"

G_DEFINE_TYPE (MwbCrashPage, mwb_crash_page, NBTK_TYPE_WIDGET)

#define CRASH_PAGE_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_CRASH_PAGE, MwbCrashPagePrivate))

enum
{
  NEW_TAB,
  RELOAD,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbCrashPagePrivate
{
  NbtkWidget   *title;
  NbtkWidget   *message;
  NbtkWidget   *page_bin;
  NbtkWidget   *new_button;
  NbtkWidget   *reload_button;
};


static void
mwb_crash_page_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_crash_page_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_crash_page_dispose (GObject *object)
{
  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (object)->priv;

  if (priv->title)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->title));
      priv->title = NULL;
    }

  if (priv->message)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->message));
      priv->message = NULL;
    }

  if (priv->page_bin)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->page_bin));
      priv->page_bin = NULL;
    }

  if (priv->new_button)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->new_button));
      priv->new_button = NULL;
    }

  if (priv->reload_button)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->reload_button));
      priv->reload_button = NULL;
    }

  G_OBJECT_CLASS (mwb_crash_page_parent_class)->dispose (object);
}

static void
mwb_crash_page_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_crash_page_parent_class)->finalize (object);
}

static void
mwb_crash_page_paint (ClutterActor *actor)
{
  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (actor)->priv;

  /* Chain up for background */
  CLUTTER_ACTOR_CLASS (mwb_crash_page_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->title));
  clutter_actor_paint (CLUTTER_ACTOR (priv->page_bin));
  clutter_actor_paint (CLUTTER_ACTOR (priv->message));
  clutter_actor_paint (CLUTTER_ACTOR (priv->new_button));
  clutter_actor_paint (CLUTTER_ACTOR (priv->reload_button));
}

static void
mwb_crash_page_pick (ClutterActor *actor, const ClutterColor *color)
{
  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (actor)->priv;

  clutter_actor_paint (CLUTTER_ACTOR (priv->new_button));
  clutter_actor_paint (CLUTTER_ACTOR (priv->reload_button));
}

static void
mwb_crash_page_allocate (ClutterActor           *actor,
                         const ClutterActorBox  *box,
                         ClutterAllocationFlags  flags)
{
  NbtkPadding padding;
  ClutterActorBox child_box;
  gfloat margin, button_width1, button_width2, width, height;

  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (actor)->priv;

#define spacing (8.f)

  /* Chain up to store box */
  CLUTTER_ACTOR_CLASS (mwb_crash_page_parent_class)->
    allocate (actor, box, flags);

  margin = (box->x2 - box->x1)/8.f;
  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  /* Allocate title */
  child_box.y1 = (gint)padding.top;
  child_box.x1 = (gint)(padding.left + margin);
  child_box.x2 = (box->x2 - box->x1) - padding.right - margin;
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->title),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &child_box.y2);
  child_box.y2 += child_box.y1;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->title), &child_box, flags);

  /* Allocate page bin */
  child_box.x2 = child_box.x1 + 400;
  child_box.y1 = (gint)(child_box.y2 + spacing);
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->page_bin),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &child_box.y2);
  child_box.y2 += child_box.y1;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->page_bin), &child_box, flags);

  /* Allocate message */
  child_box.x1 = (gint)(child_box.x2 + spacing);
  child_box.x2 = (box->x2 - box->x1) - padding.right - margin;
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->message),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &child_box.y2);
  child_box.y2 += child_box.y1;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->message), &child_box, flags);

  /* Allocate for buttons */
  width = (child_box.x2 - child_box.x1) / 2.f;
  clutter_actor_get_preferred_width (CLUTTER_ACTOR (priv->new_button),
                                     -1,
                                     NULL,
                                     &button_width1);
  clutter_actor_get_preferred_width (CLUTTER_ACTOR (priv->reload_button),
                                     -1,
                                     NULL,
                                     &button_width2);
  width = MIN (width, MAX (button_width1, button_width2));

  child_box.x1 = (gint)(child_box.x2 - width);
  child_box.y1 = (gint)(child_box.y2 + (spacing * 2));
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->new_button),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &height);
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->new_button),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &child_box.y2);
  child_box.y2 = (gint)(MAX (height, child_box.y2) + child_box.y1);

  clutter_actor_allocate (CLUTTER_ACTOR (priv->new_button), &child_box, flags);

  child_box.x1 = (gint)(child_box.x1 - width - spacing);
  child_box.x2 = (gint)(child_box.x1 + width);
  clutter_actor_allocate (CLUTTER_ACTOR (priv->reload_button),
                          &child_box,
                          flags);
}

static void
mwb_crash_page_map (ClutterActor *actor)
{
  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_crash_page_parent_class)->map (actor);

  clutter_actor_map (CLUTTER_ACTOR (priv->title));
  clutter_actor_map (CLUTTER_ACTOR (priv->page_bin));
  clutter_actor_map (CLUTTER_ACTOR (priv->message));
  clutter_actor_map (CLUTTER_ACTOR (priv->new_button));
  clutter_actor_map (CLUTTER_ACTOR (priv->reload_button));
}

static void
mwb_crash_page_unmap (ClutterActor *actor)
{
  MwbCrashPagePrivate *priv = MWB_CRASH_PAGE (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_crash_page_parent_class)->unmap (actor);

  clutter_actor_unmap (CLUTTER_ACTOR (priv->title));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->page_bin));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->message));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->new_button));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->reload_button));
}

static void
mwb_crash_page_class_init (MwbCrashPageClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbCrashPagePrivate));

  object_class->get_property = mwb_crash_page_get_property;
  object_class->set_property = mwb_crash_page_set_property;
  object_class->dispose = mwb_crash_page_dispose;
  object_class->finalize = mwb_crash_page_finalize;

  actor_class->paint = mwb_crash_page_paint;
  actor_class->pick = mwb_crash_page_pick;
  actor_class->allocate = mwb_crash_page_allocate;
  actor_class->map = mwb_crash_page_map;
  actor_class->unmap = mwb_crash_page_unmap;

  signals[NEW_TAB] =
    g_signal_new ("new-tab",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbCrashPageClass, new_tab),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  signals[RELOAD] =
    g_signal_new ("reload",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbCrashPageClass, reload),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
mwb_crash_page_new_cb (NbtkButton   *button,
                       MwbCrashPage *self)
{
  g_signal_emit (self, signals[NEW_TAB], 0);
}

static void
mwb_crash_page_reload_cb (NbtkButton   *button,
                          MwbCrashPage *self)
{
  g_signal_emit (self, signals[RELOAD], 0);
}

static void
mwb_crash_page_init (MwbCrashPage *self)
{
  ClutterActor *text;

  MwbCrashPagePrivate *priv = self->priv = CRASH_PAGE_PRIVATE (self);

  priv->title = nbtk_label_new (_("Oops!"));
  priv->message = nbtk_label_new ("");
  priv->page_bin = nbtk_bin_new ();
  priv->new_button = nbtk_button_new_with_label (_("New tab page"));
  priv->reload_button = nbtk_button_new_with_label (_("Reload page"));

  text = nbtk_label_get_clutter_text (NBTK_LABEL (priv->message));
  clutter_text_set_line_wrap (CLUTTER_TEXT (text), TRUE);
  clutter_text_set_line_wrap_mode (CLUTTER_TEXT (text),
                                   PANGO_WRAP_WORD_CHAR);
  clutter_text_set_use_markup (CLUTTER_TEXT (text), TRUE);

  clutter_actor_set_name (CLUTTER_ACTOR (priv->title), "title");
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->title),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_name (CLUTTER_ACTOR (priv->message), "message");
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->message),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->page_bin),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->new_button),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->reload_button),
                            CLUTTER_ACTOR (self));

  g_signal_connect (priv->new_button, "clicked",
                    G_CALLBACK (mwb_crash_page_new_cb), self);
  g_signal_connect (priv->reload_button, "clicked",
                    G_CALLBACK (mwb_crash_page_reload_cb), self);
}

NbtkWidget *
mwb_crash_page_new (void)
{
  return g_object_new (MWB_TYPE_CRASH_PAGE, NULL);
}

void
mwb_crash_page_set_page (MwbCrashPage *self, MwbPage *page)
{
  MwbCrashPagePrivate *priv = self->priv;

  if (!page)
    {
      nbtk_label_set_text (NBTK_LABEL (priv->message), "");
      nbtk_bin_set_child (NBTK_BIN (priv->page_bin), NULL);
    }
  else
    {
      gchar *message;
      ClutterActor *text;

      ClutterMozEmbed *mozembed = mwb_page_get_mozembed (page);
      ClutterActor *clone = mwb_aspect_clone_new (CLUTTER_ACTOR (mozembed));

      nbtk_bin_set_child (NBTK_BIN (priv->page_bin), clone);

      message =
        g_markup_printf_escaped (_("Sorry, <span color=\"#909090\">%s"
                                 "</span> appears to have caused this "
                                 "tab to crash. The rest of the browser is "
                                 "unharmed. You can try to reload the page "
                                 "or visit another one."),
                                 clutter_mozembed_get_location (mozembed));
      text = nbtk_label_get_clutter_text (NBTK_LABEL (priv->message));
      clutter_text_set_markup (CLUTTER_TEXT (text), message);
      g_free (message);
    }

  clutter_actor_queue_relayout (CLUTTER_ACTOR (self));
}

