# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL.EXCEPTION" in the root of this distribution
# including a special exception to use Elisa with Fluendo's plugins and
# about affiliation parameters.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>


from elisa.core.tests.resource_providers import CommonResourceTester, GenericSetup

from twisted.trial.unittest import TestCase
from elisa.core.utils import defer

from elisa.core.media_uri import MediaUri

from elisa.plugins.yesfm.yesfm_resource import YesfmResource
from elisa.plugins.yesfm.constants import API_KEY, API_PATH

from elisa.plugins.yesfm.models import ArtistsSearchResultModel, ArtistModel, \
                                       AlbumsSearchResultModel, AlbumModel, \
                                       TracksSearchResultModel, TrackModel, \
                                       PlayableModel, \
                                       GeneralSearchResultModel

from elisa.plugins.base.models.image import ImageModel

from simplejson.encoder import JSONEncoder


class DummyStream(object):
    def __init__(self, data):
        self.data = data
    def read(self):
        ret = self.data
        self.data = None
        return ret

class DummyResponse(object):
    def __init__(self, code, stream):
        self.code = code
        self.stream = DummyStream(stream)

class DummyHttpClient(object):
    def __init__(self):
        self.requests = []
        self.encoder = JSONEncoder()
        self.results = []

    def add_result(self, data):
        response = DummyResponse(200, self.encoder.encode(data))
        self.results.append(response)
    
    def close(self):
        return defer.succeed(self)

    def request_full(self, req):
        self.requests.append(req)
        try:
            return defer.succeed(self.results.pop(0))
        except IndexError:
            return defer.succeed(DummyResponse(404, ''))



class TestYesfmResource(CommonResourceTester, TestCase):

    resource_class = YesfmResource

    working_uris = ['http://api.yes.fm/v1/search/artist/madonna',
                    'http://api.yes.fm/v1/track/19',
                    'http://api.yes.fm/v1/auth/logout',
                    'http://cdn.yes.fm/yesfm/ondemand', # example data server
                    'http://api.yes.fm/',
                    ]
    failing_uris = ['https://api.yes.fm/v1/search/artist/madonna',
                    'http://api.yes.fm/v2']


class TestYesfmParsing(GenericSetup, TestCase):
    resource_class = YesfmResource

    def setUp(self):
        # this is a parsing test. disable the http functionality
        self.resource_class._set_session_from_login = lambda *args: None
        self.resource_class.login = lambda *args, **kw: defer.succeed(None)
        return super(TestYesfmParsing, self).setUp()

    def test_model_for_uri(self):
        to_test = [ ('auth/login', None, None), ('auth/logout', None, None),
                    ('album/7', AlbumModel, 'album'),
                    ('artist/19', ArtistModel, 'artist'),
                    ('search/madonna', GeneralSearchResultModel, 'result'),
                    ('search/artist/madonna', ArtistsSearchResultModel, None),
                    ('search/album/hefty%20five', AlbumsSearchResultModel, None),
                    ('track/1569', TrackModel, 'track'),
                    ('trackaudio/145', PlayableModel, None),
                    ('artists/A', ArtistsSearchResultModel, None),
                    ('artists/label/cooking', ArtistsSearchResultModel, None),
                    ('artists/genre/punk', ArtistsSearchResultModel, None),
                    ('search/track/madonna', TracksSearchResultModel, None),
                    ]

        for path, model_class, to_rip in to_test:
            full_path = '/%s%s' % (API_PATH, path)
            model, rip_out = self.resource.get_model_for_uri_path(full_path)
            if model_class:
                msg = "%s 's model is %s but expected a model of type %s" % \
                        (full_path, model, model_class)
                self.failUnless(isinstance(model, model_class), msg)
            else:
                msg = "Got a model for %s but expected Nothing" % full_path
                self.failIf(model, msg)

            self.assertEquals(to_rip, rip_out)

    test_model_for_uri.todo = "We don't have official support for artists/* yes"


    def test_parse_artist_search_one_item(self):
        
        data = {'totalresultsreturned' : 1,
                'result' : [ {'id' : 1,
                              'name' : 'Madonna',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' : 
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 4,
                              'score' : 98,
                                }
                            ]
                }


        def check(result_model):
            self.assertEquals(result_model.count, 1)
            self.assertEquals(len(result_model.result), 1)

            artist = result_model.result[0]
            self.failUnless(isinstance(artist, ArtistModel))

            self.assertEquals(artist.yesfmid, 1)
            self.assertEquals(artist.name, "Madonna")
            self.assertEquals(artist.radioid, 123)
            self.assertEquals(artist.radiotype, 'free')
            self.assertEquals(len(artist.images[0].references), 3)
            self.assertEquals(artist.rating, 4)
            self.assertEquals(artist.score, 98)

        model = ArtistsSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_parse_artist_search_more_items(self):
        data = {'totalresultsreturned' : 3,
                'result' : [ {'id' : 1,
                              'name' : 'Madonna',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 2,
                              'score' : 98,
                                },
                            {'id' : 2,
                              'name' : 'Coldplay',
                              'radio' : 133,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 5,
                              'score' : 78,
                                },
                            {'id' : 3000,
                              'name' : 'Bloodhound Gang',
                              'radio' : 12893,
                              'radiotype' : 'vip',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 5,
                              'score' : 80,
                                }
                            ]
                }

        def check(result_model):
            self.assertEquals(result_model.count,3)
            self.assertEquals(len(result_model.result), 3)
            self.assertEquals(['Madonna', 'Coldplay', 'Bloodhound Gang'],
                                [ i.name for i in result_model.result])

        model = ArtistsSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_parse_album_search_more_items(self):
        data = {'totalresultsreturned' : 3,
                'result' : [ {'id' : 1,
                              'title' : 'Music',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 98,
                                },
                            {'id' : 2,
                              'title' : 'X&Y',
                              'radio' : 133,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 78,
                                },
                            {'id' : 3000,
                              'title' : 'Hefty Five',
                              'radio' : 12893,
                              'radiotype' : 'vip',
                              'picture' :
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 80,
                                }
                            ]
                }

        def check(result_model):
            self.assertEquals(result_model.count, 3)
            self.assertEquals(len(result_model.result), 3)
            self.assertEquals(['Music', 'X&Y', 'Hefty Five'],
                                [ i.album for i in result_model.result])

        model = AlbumsSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_artist(self):
        data = {'id' : 108,
                'name' : 'Madonna',
                'radio' : 127,
                'radiotype' : 'vip',
                'picture' : 
                    {'small' : 'http://yay.com/viva_small.jpg',
                     'medium' :  'http://yay.com/viva_medium.jpg',
                     'large' :  'http://yay.com/viva_large.jpg',},
                'rating' : 1,
                'biography' :
                    {'text' : 'blablabla',
                     'url' : 'http://biograpyh.com/madonna',
                     'lang' : 'es'},
                'albums' : [
                            {'id' : 1,
                              'title' : 'Music',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'stats' : 
                                {'alltime': 
                                    {'playcount' : 3,
                                     'playtime' :300,
                                     'votes' : 4,
                                     'favorite' : 4,
                                    },
                                'lastmonth':
                                    {'playcount' : 1,
                                     'playtime' : 2,
                                     'votes' : 1,
                                     'favorite' : 5,
                                    },
                                 }
                                },
                            {'id' : 2,
                              'title' : 'X&Y',
                              'radio' : 133,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'stats' : 
                                {'alltime': 
                                    {'playcount' : 3,
                                     'playtime' :300,
                                     'votes' : 4,
                                     'favorite' : 4,
                                    },
                                'lastmonth':
                                    {'playcount' : 1,
                                     'playtime' : 2,
                                     'votes' : 1,
                                     'favorite' : 5,
                                    },
                                 }
                                },
                            {'id' : 3000,
                              'title' : 'Hefty Five',
                              'radio' : 12893,
                              'radiotype' : 'vip',
                              'picture' :
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'stats' : 
                                {'alltime': 
                                    {'playcount' : 3,
                                     'playtime' :300,
                                     'votes' : 4,
                                     'favorite' : 4,
                                    },
                                'lastmonth':
                                    {'playcount' : 1,
                                     'playtime' : 2,
                                     'votes' : 1,
                                     'favorite' : 5,
                                    },
                                 }
                                }
                     ],
                'relatedartists' : [
                    {'id' : 'coldplay',
                     'name' : 'coldplay',
                     'radio' : 'coldplay/radio',
                     'rating' : 24,
                     'picture' :  
                        {'small': 'http://small/',
                         'medium': 'http://medium/',
                         'large' : 'http://large'},
                    }
                ]

                }

        def check(result_model):
            self.assertEquals(result_model.name, 'Madonna' )

            pic = result_model.images[0]
            self.failUnless(isinstance(pic, ImageModel))

            self.assertEquals(len(pic.references), 3)
            self.assertEquals(
                    ['viva_small.jpg', 'viva_medium.jpg', 'viva_large.jpg'],
                    [ uri.filename for uri in pic.references])

            self.assertEquals(len(result_model.albums), 3)

            self.assertEquals(len(result_model.relatedartists), 1)

        model = ArtistModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_album(self):
        data = {'id' : 108,
                'title' : 'Viva la Vida',
                'radio' : 127,
                'radiotype' : 'vip',
                'picture' : 
                    {'small' : 'http://yay.com/viva_small.jpg',
                     'medium' :  'http://yay.com/viva_medium.jpg',
                     'large' :  'http://yay.com/viva_large.jpg',},
                'volumes' : 1,
                'publisher' : 'EMI',
                'year' : 2008,
                'artist':
                    {'id' : 90,
                     'name' : 'Coldplay'},
                'tracks' : [
                    {'id' : 1234,
                     'title' : 'Life in Technicolor',
                     'volume' : 1,
                     'trackno' : 1,
                     'lenght' : 300},
                     {'id' : 12345,
                      'title' : 'Cemetries of London',
                      'volume' : 1,
                      'trackno' : 2,
                      'length': 312
                     }
                     ]
                }

        def check(result_model):
            self.assertEquals(result_model.album, 'Viva la Vida' )

            pic = result_model.cover
            self.failUnless(isinstance(pic, ImageModel))

            self.assertEquals(len(pic.references), 3)
            self.assertEquals(
                    ['viva_small.jpg', 'viva_medium.jpg', 'viva_large.jpg'],
                    [ uri.filename for uri in pic.references])

            self.assertEquals(len(result_model.tracks), 2)

            # pick one track for testing
            track = result_model.tracks[1]
            self.failUnless(isinstance(track, TrackModel))
            self.assertEquals(track.title, 'Cemetries of London')
            self.assertEquals(track.track_number, 2)
            self.assertEquals(track.duration, 312)

            artist = result_model.artist
            self.failUnless(isinstance(artist, ArtistModel))
            self.assertEquals(artist.name, 'Coldplay')

        model = AlbumModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_parse_track_search(self):
        data = {'totalresultsreturned' : 3,
                'result' : [ {'id' : 1,
                              'title' : 'Ma',
                              'length' : 320,
                              'volume' : 1,
                              'album' : {
                                  'id' : 1234,
                                  'title' : 'music',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Madonna',
                                   },
                                },
                            },
                            {'id' : 2,
                              'title' : 'Me',
                              'length' : 30,
                              'volume' : 1,
                              'album' : {
                                  'id' : 234,
                                  'title' : 'like a virgine',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Madonna',
                                   },
                                },
                            },
                            {'id' : 4,
                              'title' : 'Mu',
                              'length' : 20,
                              'volume' : 1,
                              'album' : {
                                  'id' : 134,
                                  'title' : 'why not',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Coldplay',
                                   },
                                },
                            }

                            ]
                }

        def check(result_model):
            self.assertEquals(result_model.count,3)
            self.assertEquals(len(result_model.result), 3)
            self.assertEquals(['Ma', 'Me', 'Mu'],
                                [ i.title for i in result_model.result])
            
            self.assertEquals(['Madonna', 'Madonna', 'Coldplay'],
                    [i.album.artist.name for i in result_model.result])

        model = TracksSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_general_search(self):
        data =  { 'tracks' :{
                    'totalresultsreturned' : 3,
                    'result' : [
                              {'id' : 1,
                              'title' : 'Ma',
                              'length' : 320,
                              'volume' : 1,
                              'album' : {
                                  'id' : 1234,
                                  'title' : 'music',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Madonna',
                                   },
                                },
                            },
                            {'id' : 2,
                              'title' : 'Me',
                              'length' : 30,
                              'volume' : 1,
                              'album' : {
                                  'id' : 234,
                                  'title' : 'like a virgine',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Madonna',
                                   },
                                },
                            },
                            {'id' : 4,
                              'title' : 'Mu',
                              'length' : 20,
                              'volume' : 1,
                              'album' : {
                                  'id' : 134,
                                  'title' : 'why not',
                                  'picture' :  
                                    {'small': 'http://small/',
                                     'medium': 'http://medium/',
                                     'large' : 'http://large'},
                                   'artist' : {
                                       'id' : 12345,
                                       'name': 'Coldplay',
                                   },
                                },
                            },
                           ],
                        },
                    'albums': {
                        'totalresultsreturned' : 1,
                        'result' : [ {'id' : 1,
                              'title' : 'Music',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 98,
                                },
                            {'id' : 2,
                              'title' : 'X&Y',
                              'radio' : 133,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 78,
                                },
                            {'id' : 3000,
                              'title' : 'Hefty Five',
                              'radio' : 12893,
                              'radiotype' : 'vip',
                              'picture' :
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'score' : 80,
                                }]},

                    'artists' :
                        {'totalresultsreturned' : 1,
                        'result' : [ {'id' : 1,
                              'name' : 'Madonna',
                              'radio' : 123,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 2,
                              'score' : 98,
                                },
                            {'id' : 2,
                              'name' : 'Coldplay',
                              'radio' : 133,
                              'radiotype' : 'free',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 5,
                              'score' : 78,
                                },
                            {'id' : 3000,
                              'name' : 'Bloodhound Gang',
                              'radio' : 12893,
                              'radiotype' : 'vip',
                              'picture' :  
                                {'small': 'http://small/',
                                 'medium': 'http://medium/',
                                 'large' : 'http://large'},
                              'rating' : 5,
                              'score' : 80,
                                }
                            ],},
                }

        def check(result_model):
            self.assertEquals(len(result_model.artists.result), 3)
            self.assertEquals(len(result_model.tracks.result), 3)
            self.assertEquals(len(result_model.albums.result), 3)

        model = GeneralSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

    def test_general_search_nothing_found(self):
        data =  { 'tracks' :{
                    'totalresultsreturned' : 0,
                    'result' : None },
                  'albums': {
                    'totalresultsreturned' : 0,
                    'result' : None},
                  'artists' : {
                    'totalresultsreturned' : 1,
                    'result' : None},
                }

        def check(result_model):
            self.assertEquals(len(result_model.artists.result), 0)
            self.assertEquals(len(result_model.tracks.result), 0)
            self.assertEquals(len(result_model.albums.result), 0)

        model = GeneralSearchResultModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr



    def test_playable(self):
        data = {'success' : True,
                'audiouri' : 'http://example.com/test.mp3'
                }

        def check(result_model):
            ref_uri = MediaUri('http://example.com/test.mp3')
            self.assertEquals(result_model.uri, ref_uri)

        model = PlayableModel()
        dfr = self.resource.parse_data_into_model(data, model)
        dfr.addCallback(check)
        return dfr

class TestYesfmLogin(GenericSetup, TestCase):
    resource_class = YesfmResource
    empty_login_dict = {'userkey' : ''}

    def setUp(self):
        # overwrite the login so that it can be tested in a known environment
        self.login = self.resource_class.login
        self.resource_class._set_session_from_login = lambda x, **kw: None
        self.resource_class.login = lambda x, userkey=None: defer.succeed(None)

        def set_http_client(resource):
            self.http_client = DummyHttpClient()
            self.resource._http_client = self.http_client

        dfr = super(TestYesfmLogin, self).setUp()
        dfr.addCallback(set_http_client)
        return dfr

    def tearDown(self):
        self.resource_class.login = self.login
        return super(TestYesfmLogin, self).tearDown()

    def test_non_auth(self):
        # setup for this test
        self.http_client.add_result( {'success' : 'true', 'session':\
                                      {'sessionkey': 'pocahontas',
                                       'userkey' : 'meeko',
                                       'usertype' : 'anon'} } )

        def check(result):
            self.assertEquals(len(self.http_client.requests), 1)

            self.check_request(self.http_client.requests[0])

            self.assertEquals(result.sessionkey, 'pocahontas')

        dfr = self.login(self.resource)
        dfr.addCallback(check)
        return dfr

    def test_user_auth(self):
        # there are user and password specified in the config, so they should be
        # used for login

        self.http_client.add_result( {'success' : 'true', 'session':\
                                      {'sessionkey': 'john smith',
                                       'userkey' : '42',
                                       'usertype' : 'reg'} } )

        def check(result):
            self.assertEquals(len(self.http_client.requests), 1)

            self.check_request(self.http_client.requests[0])

            self.assertEquals(result.sessionkey, 'john smith')
            self.assertEquals(result.userkey, '42')

        dfr = self.login(self.resource, user='pocahontas', password='disney')
        dfr.addCallback(check)
        return dfr
        
    def test_userkey(self):
        self.http_client.add_result( {'success' : 'true', 'session':\
                                      {'sessionkey': 'smith',
                                       'userkey' : '33',
                                       'usertype' : 'reg'} } )

        def check(result):
            self.assertEquals(len(self.http_client.requests), 1)

            self.check_request(self.http_client.requests[0])

            self.assertEquals(result.sessionkey, 'smith')
            self.assertEquals(result.userkey, '33')

        dfr = self.login(self.resource, userkey='wingapo')
        dfr.addCallback(check)
        return dfr
        
    def test_userkey_missing_values(self):
        """
        It seems we sometimes receive a response that is not exactly how the
        API states it: it is missing some fields. This test ensure that the
        login still works.
        """
        self.http_client.add_result( {'success' : 'true', 'session':\
                                      {'sessionkey': 'smith'} } )

        def check(result):
            self.assertEquals(len(self.http_client.requests), 1)

            self.check_request(self.http_client.requests[0])

            self.assertEquals(result.sessionkey, 'smith')

        dfr = self.login(self.resource, userkey='wingapo')
        dfr.addCallback(check)
        return dfr


    def check_request(self, request):

        # check for the API key
        headers = request.headers
        msg = "Missing the API key"
        self.assertEquals(headers.getRawHeaders('X-yesfm-APIkey')[0], API_KEY, msg)

class TestYesfmPost(GenericSetup, TestCase):
    resource_class = YesfmResource

    def setUp(self):
        # this is a parsing test. disable the http functionality
        self.resource_class._set_session_from_login = lambda x: None
        self.resource_class.login = lambda x, userkey=None: defer.succeed(None)
        return super(TestYesfmPost, self).setUp()

    def test_post_session_key(self):
        self.resource.session_key = None

        def check(result):
            self.assertEquals(self.resource.session_key, 'test')

        uri = MediaUri('http://api.yes.fm/')

        dfr = self.resource.post(uri, sessionkey='test')
        dfr.addCallback(check)
        return dfr

    def test_login(self):

        def login(**params):
            return defer.succeed(params)
        
        self.resource.login = login

        def check(result):
            self.assertEquals(result['user'], 'superuser')
            self.assertEquals(result['password'], 'elisa')

        uri = MediaUri('http://api.yes.fm/v1/auth/login')
        dfr = self.resource.post(uri, user='superuser', password='elisa')
        dfr.addCallback(check)
        return dfr

    def test_post_userkey(self):
        self.resource.config['userkey'] = None

        def check(result):
            self.assertEquals(self.resource.config['userkey'], 'test')

        uri = MediaUri('http://api.yes.fm/')

        dfr = self.resource.post(uri, userkey='test')
        dfr.addCallback(check)
        return dfr 
