# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL.EXCEPTION" in the root of this distribution
# including a special exception to use Elisa with Fluendo's plugins and
# about affiliation parameters.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.components.model import Model
from elisa.core import common
from elisa.core.utils import defer

from elisa.plugins.base.models.media import PlayableModel as BasePlayabelModel
from elisa.plugins.base.models.audio import AlbumModel as BaseAlbumModel,\
                                            ArtistModel as BaseArtistModel, \
                                            TrackModel as BaseTrackModel

from elisa.core.media_uri import MediaUri
from elisa.plugins.yesfm.constants import SERVER_NAME, API_PATH


class ArtistModel(BaseArtistModel):
    def __init__(self):
        super(ArtistModel, self).__init__()
        self.yesfmid = 0
        self.albums = []
        self.relatedartists =[]

    # FIXME: this property-hacks belong into the parsing system
    def picture__set(self, value):
        self.images.append(value)

    picture = property(fset=picture__set)

class PlayableModel(BasePlayabelModel):

    # FIXME: this property-hacks belong into the parsing system
    def yesfm_uri__set(self, uri):
        self.uri = MediaUri(uri)

    yesfm_uri = property(fset=yesfm_uri__set)

class TrackModel(BaseTrackModel):
    def __init__(self):
        super(TrackModel, self).__init__()
        self.yesfmid = 0

    def playable_uri__get(self):
        return MediaUri('http://%s/%strackaudio/%s' %
                (SERVER_NAME, API_PATH, self.yesfmid) )

    def playable_uri__set(self, uri):
        # we ignore whatever is set here because we don't care. But anyway for
        # API compability we need to have a setter as the BaseClass is setting
        # this variable on __init__
        pass

    playable_uri = property(fget=playable_uri__get, fset=playable_uri__set)

    def get_playable_model(self):
        def add_title(model):
            model.title = self.title
            return model

        uri = self.playable_uri
        model, dfr = common.application.resource_manager.get(uri)
        dfr.addCallback(add_title)
        return dfr

    def get_artists(self):
        if self.artist != None:
            return defer.succeed([self.artist])
        else:
            def update_artist(model):
                self.artist = model.album.artist.name
                return [self.artist]

            uri = MediaUri("http://api.yes.fm/v1/track/%s" % self.yesfmid)
            model, dfr = common.application.resource_manager.get(uri)
            dfr.addCallback(update_artist)
            return dfr

class AlbumModel(BaseAlbumModel):
    def __init__(self):
        super(AlbumModel, self).__init__()
        self.yesfmid = 0

    # FIXME: this property-hacks belong into the parsing system
    def picture__set(self, value):
        self.cover = value

    def picture__get(self):
        return self.cover

    def get_tracks(self):
        if len(self.tracks) > 0:
            return defer.succeed(self.tracks)
        else:
            def update_tracks(model):
                for track in model.tracks:
                    track.artist = self.artist.name
                    self.tracks.append(track)
                return self.tracks

            uri = MediaUri("http://api.yes.fm/v1/album/%s" % self.yesfmid)
            model, dfr = common.application.resource_manager.get(uri)
            dfr.addCallback(update_tracks)
            return dfr

    picture = property(fget=picture__get, fset=picture__set)

    def name__get(self):
        # FIXME: AlbumModel defines 'album' as containing the name of the
        # album which is not a good API. To keep sanity is Yes.fm's code
        # 'name' is a proxy to 'album' until mainstream API is fixed.
        return self.album

    name = property(fget=name__get)

    def get_artist_name(self):
        return defer.succeed(self.artist.name)

# Search Result Models

# general class implementing the behaviour


class SearchResultModel(Model):
    """
    A model holding the result of a Search Request to YesFm.

    @ivar count:  the total number of result that have been found
    @type count:  c{int}
    @type result: c{list}
    """
    def __init__(self):
        super(SearchResultModel, self).__init__()
        self.count = 0
        self.result = []

class ArtistsSearchResultModel(SearchResultModel):
    pass

class AlbumsSearchResultModel(SearchResultModel):
    pass

class TracksSearchResultModel(SearchResultModel):
    pass

class GeneralSearchResultModel(Model):
    # FIXME: add some DOCS
    def __init__(self):
        super(GeneralSearchResultModel, self).__init__()
        # always their corresponding SearchResultModels
        self.artists = []
        self.albums = []
        self.tracks = []
        # not yet supported
        self.radios = []
        self.users = []

class LoginModel(Model):
    def __init__(self):
        super(LoginModel, self).__init__()
        self.success = False
        self.sessionkey = None
        self.userkey = None
        self.usertype = None
