# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
models with (in)direct database access
"""

from elisa.extern.storm_wrapper import wrapper

from storm.locals import Unicode, Int, DateTime, Bool

from elisa.core.media_uri import MediaUri, quote
from elisa.core.utils import defer

from elisa.plugins.base.models.media import PlayableModel


class MusicAlbum(object):
    """
    Represents one album
    """
    __storm_table__ = "music_albums"
    name = Unicode(primary=True)
    # URI of a picture file for the album's cover
    # eg: http://ecx.images-amazon.com/images/I/510D51P8YKL._SL500_.jpg
    cover_uri = Unicode()
    release_date = DateTime()

    def get_tracks(self):
        dfr = self.tracks.find()
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(lambda items: sorted(items, key=lambda a: a.file_path.lower()))
        return dfr

    def get_artist_name(self):
        """
        Retrieve the artist name associated with the first track of the album.
        If there's no track or no artist is found, empty string is returned.
        """

        def got_tracks(tracks):
            def got_artists(artists):
                if len(artists):
                    artist_name = artists[0]
                else:
                    artist_name = ''
                return artist_name

            if len(tracks) > 0:
                dfr = tracks[0].artists.values(Artist.name)
                dfr.addCallback(got_artists)
            else:
                dfr = defer.succeed('')
            return dfr

        dfr = self.tracks.find()
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(got_tracks)
        return dfr

class PhotoAlbum(object):
    __storm_table__ = "photo_albums"
    name = Unicode(primary=True)
    preview_uri = Unicode() # a uri to a generated preview 

class Artist(object):
    """
    Represents an artist
    """
    __storm_table__ = 'artists'
    # FIXME: this class is not very precise
    name = Unicode(primary=True)
    # URI of a picture file that represents the artist
    # eg: http://www.discogs.com/image/A-56260-1189986364.jpeg
    image_uri = Unicode()

    def get_tracks(self):
        dfr = self.tracks.find()
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(lambda items: sorted(items, key=lambda a: a.file_path.lower()))
        return dfr

class TrackArtist(object):
    """
    weak table to map artists to tracks
    """
    __storm_table__ = 'track_artists'
    __storm_primary__ = "track_path", "artist_name"

    artist_name = Unicode()
    track_path = Unicode()


class MusicTrack(object):
    """
    Represents one music track
    """
    __storm_table__ = 'music_tracks'
    # and exactly one file
    file_path = Unicode(primary=True)
    # it has a title
    title = Unicode()
    # track number
    track_number = Int()
    # duration
    duration = Int()
    # and is exactly in one album
    album_name = Unicode()
    # genre
    genre = Unicode()

    def get_album(self):
        """
        Return a deferred that will return the album containing the track as an
        instance of L{elisa.plugins.base.models.audio.AlbumModel}.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return self.album

    def get_artists(self):
        """
        Return a deferred that will return the list of artist names for the
        track as C{unicode} instances.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        def values(result):
            return result.values(Artist.name)

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        deferred = self.artists.find()
        deferred.addCallback(set_distinct)
        deferred.addCallback(values)
        return deferred

    def get_playable_model(self):
        """
        Return a deferred that will return an instance of
        L{elisa.plugins.base.models.media.PlayableModel} for the track.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        model = PlayableModel()
        model.uri = MediaUri('file://' + self.file_path)
        model.title = self.title
        return defer.succeed(model)

class Image(object):
    """
    represent an image
    """
    __storm_table__ = 'images'
    file_path = Unicode(primary=True)
    size = Unicode()
    # the time when the image was shot
    shot_time = DateTime()
    # is it done with flash
    with_flash = Bool()
    # orientation of the image
    orientation = Int()

    # GPS informations: where was the image made?
    gps_altitude = Int()
    gps_latitude = Unicode()
    gps_longitude = Unicode()

    # and is exactly in one album
    album_name = Unicode()

class Video(object):
    """
    represents an video
    """
    __storm_table__ = 'videos'
    file_path = Unicode(primary=True)
    duration = Int()
    size = Unicode()
    codec = Unicode()
    thumbnail_uri = Unicode()


class File(object):
    """
    represente a file
    """
    __storm_table__ = 'files'
    path = Unicode(primary=True)
    source = Unicode()
    mime_type = Unicode()
    modification_time = Int()
    deleted = Int()
    playcount = Int()
    last_played = Int()


class Tag(object):
    """
    representation of a TAG
    """
    __storm_table__ = "tags"
    name = Unicode(primary=True)


class FileTags(object):
    """
    map multiple tags to multiple Files
    """
    __storm_table__ = 'file_tags'
    __storm_primary__ = "file_path", "tag_name"

    tag_name = Unicode()
    file_path = Unicode()


# links for music/audio
Artist.tracks= wrapper.DeferredReferenceSet(Artist.name,
        TrackArtist.artist_name,
        TrackArtist.track_path,
        MusicTrack.file_path)

MusicTrack.album = wrapper.DeferredReference(MusicTrack.album_name,
        MusicAlbum.name)
MusicTrack.artists = wrapper.DeferredReferenceSet(MusicTrack.file_path,
        TrackArtist.track_path, TrackArtist.artist_name, Artist.name)
MusicAlbum.tracks = wrapper.DeferredReferenceSet(MusicAlbum.name,
        MusicTrack.album_name)

# links for image/photo
Image.album = wrapper.DeferredReference(Image.album_name, PhotoAlbum.name)
PhotoAlbum.photos = wrapper.DeferredReferenceSet(PhotoAlbum.name, Image.album_name)

# link types to file
MusicTrack.file = wrapper.DeferredReference(MusicTrack.file_path, File.path)
Image.file = wrapper.DeferredReference(Image.file_path, File.path)
Video.file = wrapper.DeferredReference(Video.file_path, File.path)

# link file to more specific types
File.music_track = wrapper.DeferredReference(File.path, MusicTrack.file_path)
File.image = wrapper.DeferredReference(File.path, Image.file_path)
File.tags = wrapper.DeferredReferenceSet(File.path, FileTags.file_path,
        FileTags.tag_name, Tag.name)

Tag.files = wrapper.DeferredReferenceSet(Tag.name, FileTags.tag_name,
        FileTags.file_path, File.path)
