/*
* Copyright (C) 2008  Intel Corporation
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License, version 2, as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
* In addition, as a special exception, Intel gives permission to link
* the code of portions of this program with the OpenSSL project's
* "OpenSSL" library (or with modified versions of it that use the same
* license as the "OpenSSL" library), and distribute the linked
* executables.  You must obey the GNU General Public License in all
* respects for all of the code used other than "OpenSSL".  If you modify
* this file, you may extend this exception to your version of the file,
* but you are not obligated to do so.  If you do not wish to do so,
* delete this exception statement from your version.
*/

#ifndef TABLET_HELPER_H
#define TABLET_HELPER_H

#ifdef WIN32
#include <windows.h>
#else
#include <pthread.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <set>
#endif /** WIN32 */

#include "plugins/DcsPlugin.h"
#include "plugins/tablet/TabletDef.h"
#include "plugins/tablet/TabletPlugin.h"

/**
 * @class TabletHelper
 * @brief TabletHelper provides help on operating Tablet driver.
 *
 * It is implemented as a singleton.
 */

class TabletHelper
{
public:
	static TabletHelper &GetInstance()
	{
		static TabletHelper helper;
		return helper;
	};

	/** public functions which really implement DCS Plugin interfface. */
	int Initialize(DCS_Funcs *pDispatcher);
	DCS_Return_Code HandleRequest(DCS_RequestData *pRequest);
	
	int Cleanup();

private:

#ifdef WIN32	
	TabletHelper(void)
		: m_pFuncs(NULL), m_hTabletEvent(NULL), m_hExitEvent(NULL),
		m_hThread(NULL), m_hDevice(NULL)
	{
	};
	
	~TabletHelper(void)
	{
	};
#else
	TabletHelper(void)
		: m_Thread(0), m_Device(-1)
	{
	};

	~TabletHelper(void)
	{
	};
#endif /** WIN32 */

	/** tablet related functions */
	BOOL StartTablet();
	BOOL StopTablet();
	BOOL GetTabletData(DCS_Tablet_Sensor_Data &data);
	
	/** declare copy constructor and equal operator to prevent
	 *  copied instance
	 */
	TabletHelper(const TabletHelper &);
	TabletHelper &operator =(const TabletHelper &);

	/** pointer to DCS dispatch function */
	DCS_Funcs *m_pFuncs;

	static const char * const MODULE_NAME; /** tablet chip module name */

#ifdef WIN32
	/** WIN32 working thread function */
	friend DWORD WINAPI TabletEventHandler(LPVOID lpParameter);

	void CloseHandles();

	static const wchar_t * const EVENT_NAME; /** event shared with driver */
	static const wchar_t * const DEVICE_PATH; /** path to open driver */

	HANDLE m_hTabletEvent; /** event to get notification from driver */
	HANDLE m_hExitEvent; /** event to exit working thread */
	HANDLE m_hThread; /** working thread */
	HANDLE m_hDevice; /** tablet device */

#else
	/** Linux working thread function */
	friend void *TabletEventHandler(void *arg);	
        /* user ID related functions to record which user disabled
 *          * system orientation notifications
 *                   */
        void AddUserId(const int &userId)
        {
                LockHelper helper(m_UserIdLock);
                m_ClientIdSet.insert(userId);
        };

        void EmptyUserId()
        {
                LockHelper helper(m_UserIdLock);
                m_ClientIdSet.clear();
        };

        void RemoveUserId(const int &userId)
        {
                LockHelper helper(m_UserIdLock);
                m_ClientIdSet.erase(userId);
        };

        BOOL IsUserIdEmpty()
        {
                LockHelper helper(m_UserIdLock);
                return m_ClientIdSet.empty();
        };
        pthread_mutex_t m_UserIdLock; /** lock to protect user ID set */

        /* set to record the user IDs    */
        std::set<int> m_ClientIdSet;

	static const char * const DEVICE_PATH; /** path to open driver */

	pthread_t m_Thread;
	int m_Device;
#endif /** WIN32 */

	
	/**
	 * @class LockHelper
	 * @brief LcokHelper provides auto lock and unlock approach by
	 *        constructor and destructor.
	 */

#ifdef WIN32
	class LockHelper
	{
	public:
		LockHelper(CRITICAL_SECTION &lock)
			: m_Lock(lock)
		{
			EnterCriticalSection(&m_Lock);
		};

		~LockHelper()
		{
			LeaveCriticalSection(&m_Lock);
		};

	private:
		CRITICAL_SECTION &m_Lock;
	};
#else
	class LockHelper
	{
	public:
		LockHelper(pthread_mutex_t &lock)
			:m_Lock(lock)
		{
			pthread_mutex_lock(&m_Lock);
		};

		~LockHelper()
		{
			pthread_mutex_unlock(&m_Lock);
		};

	private:
		pthread_mutex_t m_Lock;
	};
#endif /** WIN32 */
};
#endif /** TABLET_HELPER_H */

